<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\GuruModel;
use App\Models\KelasModel;
use App\Models\MapelModel;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet; // Tambahkan ini
use PhpOffice\PhpSpreadsheet\Writer\Xlsx; // Tambahkan ini

class Guru extends BaseController
{
    // ... existing code (construct, index, create, store, edit, update, delete) ...
    protected $guruModel;
    protected $kelasModel;
    protected $mapelModel;
    protected $db;

    public function __construct()
    {
        $this->guruModel = new GuruModel();
        $this->kelasModel = new KelasModel();
        $this->mapelModel = new MapelModel();
        $this->db = \Config\Database::connect();
    }
    
    // ... method index, create, store, edit, update, delete yang sudah ada ...
    // Pastikan method delete ada sebelum import
    
    public function index()
    {
        $data = [
            'title' => 'Data Guru',
            'guru' => $this->guruModel->findAll()
        ];
        return view('admin/guru/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Guru',
            'kelas' => $this->kelasModel->findAll(),
            'mapel' => $this->mapelModel->findAll(),
            'validation' => \Config\Services::validation()
        ];
        return view('admin/guru/create', $data);
    }

    public function store()
    {
        if (!$this->validate([
            'nip' => 'required',
            'nama_lengkap' => 'required',
            'username' => 'required|is_unique[guru.username]',
            'password' => 'required|min_length[6]',
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png]'
        ])) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fotoName = 'default.jpg';
        $fileFoto = $this->request->getFile('foto');
        if ($fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move('uploads/profil', $fotoName);
        }

        $dataGuru = [
            'nip' => $this->request->getPost('nip'),
            'nama_lengkap' => $this->request->getPost('nama_lengkap'),
            'username' => $this->request->getPost('username'),
            'password' => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'foto' => $fotoName
        ];

        $this->guruModel->insert($dataGuru);
        $guruId = $this->guruModel->getInsertID();

        $kelasIds = $this->request->getPost('kelas_id');
        if ($kelasIds) {
            $dataKelas = [];
            foreach ($kelasIds as $kid) {
                $dataKelas[] = ['guru_id' => $guruId, 'kelas_id' => $kid];
            }
            $this->db->table('guru_kelas')->insertBatch($dataKelas);
        }

        $mapelIds = $this->request->getPost('mapel_id');
        if ($mapelIds) {
            $dataMapel = [];
            foreach ($mapelIds as $mid) {
                $dataMapel[] = ['guru_id' => $guruId, 'mapel_id' => $mid];
            }
            $this->db->table('guru_mapel')->insertBatch($dataMapel);
        }

        return redirect()->to('admin/guru')->with('success', 'Data Guru berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $guru = $this->guruModel->find($id);
        if (!$guru) {
            return redirect()->to('admin/guru')->with('error', 'Data tidak ditemukan.');
        }

        $guruKelas = $this->db->table('guru_kelas')->where('guru_id', $id)->get()->getResultArray();
        $guruMapel = $this->db->table('guru_mapel')->where('guru_id', $id)->get()->getResultArray();

        $data = [
            'title' => 'Edit Guru',
            'guru' => $guru,
            'kelas' => $this->kelasModel->findAll(),
            'mapel' => $this->mapelModel->findAll(),
            'selected_kelas' => array_column($guruKelas, 'kelas_id'),
            'selected_mapel' => array_column($guruMapel, 'mapel_id'),
            'validation' => \Config\Services::validation()
        ];
        return view('admin/guru/edit', $data);
    }

    public function update($id)
    {
        if (!$this->validate([
            'nip' => 'required',
            'nama_lengkap' => 'required',
            'username' => "required|is_unique[guru.username,id,$id]",
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png]'
        ])) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $guru = $this->guruModel->find($id);
        $fotoName = $guru['foto'];
        
        $fileFoto = $this->request->getFile('foto');
        if ($fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move('uploads/profil', $fotoName);
            if ($guru['foto'] != 'default.jpg' && file_exists('uploads/profil/' . $guru['foto'])) {
                unlink('uploads/profil/' . $guru['foto']);
            }
        }

        $dataUpdate = [
            'nip' => $this->request->getPost('nip'),
            'nama_lengkap' => $this->request->getPost('nama_lengkap'),
            'username' => $this->request->getPost('username'),
            'foto' => $fotoName
        ];

        if ($this->request->getPost('password')) {
            $dataUpdate['password'] = password_hash($this->request->getPost('password'), PASSWORD_DEFAULT);
        }

        $this->guruModel->update($id, $dataUpdate);

        $this->db->table('guru_kelas')->where('guru_id', $id)->delete();
        $kelasIds = $this->request->getPost('kelas_id');
        if ($kelasIds) {
            $dataKelas = [];
            foreach ($kelasIds as $kid) {
                $dataKelas[] = ['guru_id' => $id, 'kelas_id' => $kid];
            }
            $this->db->table('guru_kelas')->insertBatch($dataKelas);
        }

        $this->db->table('guru_mapel')->where('guru_id', $id)->delete();
        $mapelIds = $this->request->getPost('mapel_id');
        if ($mapelIds) {
            $dataMapel = [];
            foreach ($mapelIds as $mid) {
                $dataMapel[] = ['guru_id' => $id, 'mapel_id' => $mid];
            }
            $this->db->table('guru_mapel')->insertBatch($dataMapel);
        }

        return redirect()->to('admin/guru')->with('success', 'Data Guru berhasil diperbarui.');
    }

    public function delete($id)
    {
        $guru = $this->guruModel->find($id);
        if ($guru['foto'] != 'default.jpg' && file_exists('uploads/profil/' . $guru['foto'])) {
            unlink('uploads/profil/' . $guru['foto']);
        }
        
        $this->guruModel->delete($id);
        return redirect()->to('admin/guru')->with('success', 'Data Guru berhasil dihapus.');
    }

    public function import()
    {
        $file = $this->request->getFile('file_excel');

        if ($file && $file->isValid() && !$file->hasMoved()) {
            $ext = $file->getClientExtension();
            
            if (!in_array($ext, ['xls', 'xlsx'])) {
                return redirect()->back()->with('error', 'Format file harus .xls atau .xlsx');
            }

            if (!class_exists(\PhpOffice\PhpSpreadsheet\IOFactory::class)) {
                return redirect()->back()->with('error', 'Library PhpSpreadsheet belum terinstall.');
            }

            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $data = $sheet->toArray();

                $insertData = [];
                
                foreach ($data as $key => $row) {
                    if ($key == 0) continue; 

                    if (empty($row[1]) || empty($row[2])) continue;

                    $insertData[] = [
                        'nip'          => $row[0] ?? '-',
                        'nama_lengkap' => $row[1],
                        'username'     => $row[2],
                        'password'     => password_hash((string)$row[3], PASSWORD_DEFAULT),
                        'foto'         => 'default.jpg',
                        'created_at'   => date('Y-m-d H:i:s')
                    ];
                }

                if (!empty($insertData)) {
                    $this->guruModel->insertBatch($insertData);
                    return redirect()->to('admin/guru')->with('success', count($insertData) . ' Data guru berhasil diimport.');
                }

                return redirect()->back()->with('error', 'Tidak ada data yang dapat dibaca dari file.');

            } catch (\Exception $e) {
                return redirect()->back()->with('error', 'Gagal import: ' . $e->getMessage());
            }
        }

        return redirect()->back()->with('error', 'Gagal mengupload file.');
    }

    // --- FITUR DOWNLOAD TEMPLATE ---
    public function downloadTemplate()
    {
        if (!class_exists(\PhpOffice\PhpSpreadsheet\Spreadsheet::class)) {
            return redirect()->back()->with('error', 'Library PhpSpreadsheet belum terinstall.');
        }

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $sheet->setCellValue('A1', 'NIP');
        $sheet->setCellValue('B1', 'Nama Lengkap (Wajib)');
        $sheet->setCellValue('C1', 'Username (Wajib & Unik)');
        $sheet->setCellValue('D1', 'Password');

        // Style Header (Bold & Background Kuning)
        $headerStyle = [
            'font' => ['bold' => true],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['argb' => 'FFFFFF00'],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                ],
            ],
        ];
        $sheet->getStyle('A1:D1')->applyFromArray($headerStyle);

        // Contoh Data
        $sheet->setCellValue('A2', '12345678');
        $sheet->setCellValue('B2', 'Budi Santoso');
        $sheet->setCellValue('C2', 'budis');
        $sheet->setCellValue('D2', '123456');

        // Auto Width
        foreach(range('A','D') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'template_import_guru.xlsx';

        // Set Header untuk Download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="'. $filename .'"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }
}